-- 1. 资源声明：告诉游戏这个建筑需要用到的视觉和声音资源。
local assets = {
    Asset("ANIM", "anim/potted_boardleaf.zip"), -- 指向建筑的动画文件
    -- Asset("SOUND", "sound/your_structure.fsb"), -- 如果需要音效，可以取消注释
}

-- 2. 建筑的构建函数。这是定义建筑核心行为的地方。
local function fn()
    
    local inst = CreateEntity()

    inst.entity:AddTransform() -- 让建筑有位置、旋转、缩放
    inst.entity:AddAnimState() -- 让建筑可以播放动画
    inst.entity:AddNetwork() -- 让建筑在联机模式下同步
    inst.entity:AddSoundEmitter() -- 让建筑可以播放声音

    -- 设置建筑的物理碰撞（使其可以被角色和其他实体碰撞）
    MakeObstaclePhysics(inst, 0.5) -- 参数1.0是碰撞半径，根据你的建筑大小调整

    -- 动画状态设置
    inst.AnimState:SetBank("entity") -- 动画银行名，对应动画文件中的设定
    inst.AnimState:SetBuild("potted_boardleaf") -- 动画构建名，通常与Bank相同
    inst.AnimState:PlayAnimation("potted_boardleaf_idle",true) -- 默认播放的动画，例如"idle"（闲置状态）
    inst.AnimState:SetLightOverride(1)
    inst.AnimState:SetFinalOffset(1) -- 确保渲染在正确层级

    local function OnSpawnFn(inst)
        inst.AnimState:PlayAnimation("potted_boardleaf_place")
        -- 在放置动画播放完后自动回到闲置状态
        inst:DoTaskInTime(inst.AnimState:GetCurrentAnimationLength(), function()
            inst.AnimState:PlayAnimation("potted_boardleaf_idle", true)
        end)
    end

    -- 如果希望建筑能被玩家检查，添加可检查组件并设置检查语句
    inst:AddComponent("inspectable")
    inst.components.inspectable.nameoverride = "POTTED_BOARDLEAF" -- 可选，使用特定的检查名

    -- 确保客户端和服务端数据同步
    inst.entity:SetPristine()
    if not TheWorld.ismastersim then
        return inst
    end

    -- ######## 以下代码只在服务器上运行 ########

    -- 修复燃烧组件 - 添加完整的燃烧配置
    inst:AddComponent("burnable")
    
    -- 重要：设置燃烧特效
    inst.components.burnable:SetOnIgniteFn(function(inst)
        -- 确保有燃烧音效
        inst.SoundEmitter:PlaySound("dontstarve/common/fireAddFuel")
        
        -- 添加多个燃烧特效来创建更大的火焰
        if not inst.fire_fx then
            inst.fire_fx = {}
            
            -- 主火焰 - 放大2倍
            local main_fire = SpawnPrefab("campfirefire")
            if main_fire then
                main_fire.entity:SetParent(inst.entity)
                main_fire.Transform:SetPosition(0, 1, 0) -- 提高位置
                -- 放大火焰特效
                main_fire.AnimState:SetScale(2, 2, 2)
                table.insert(inst.fire_fx, main_fire)
            end
            
            -- 副火焰 - 增加火焰体积
            local sub_fire1 = SpawnPrefab("fire")
            if sub_fire1 then
                sub_fire1.entity:SetParent(inst.entity)
                sub_fire1.Transform:SetPosition(0.3, 0.8, 0.2)
                sub_fire1.AnimState:SetScale(1.5, 1.5, 1.5)
                table.insert(inst.fire_fx, sub_fire1)
            end
            
            local sub_fire2 = SpawnPrefab("fire")
            if sub_fire2 then
                sub_fire2.entity:SetParent(inst.entity)
                sub_fire2.Transform:SetPosition(-0.3, 0.6, -0.2)
                sub_fire2.AnimState:SetScale(1.5, 1.5, 1.5)
                table.insert(inst.fire_fx, sub_fire2)
            end
            
            -- 添加火焰粒子效果
            local fire_fx = SpawnPrefab("firesplash_fx")
            if fire_fx then
                fire_fx.entity:SetParent(inst.entity)
                fire_fx.Transform:SetPosition(0, 0.5, 0)
                table.insert(inst.fire_fx, fire_fx)
            end
        end
        
        -- 开始持续伤害
        if not inst.burn_task then
            inst.sound_counter = 0 -- 确保在任务开始前初始化计数器
            inst.burn_task = inst:DoPeriodicTask(0.5, function() -- 改为每0.5秒一次
                -- 确保sound_counter不为nil
                if inst.sound_counter == nil then
                    inst.sound_counter = 0
                end
                
                if inst.components.burnable and inst.components.burnable:IsBurning() then
                    local x, y, z = inst.Transform:GetWorldPosition()
                    local ents = TheSim:FindEntities(x, y, z, 3, nil, {"playerghost", "ghost", "INLIMBO", "FX"}) -- 扩大范围到3
                    
                    for i, ent in ipairs(ents) do
                        if ent.components.health and not ent.components.health:IsDead() and ent:HasTag("player") then
                            -- 更频繁的伤害，但单次伤害降低以平衡
                            ent.components.health:DoFireDamage(2) -- 每0.5秒2点伤害
                            
                            -- 更频繁的温度上升
                            if ent.components.temperature then
                                ent.components.temperature:SetTemperature(ent.components.temperature:GetCurrent() + 1)
                                
                                -- 强制触发屏幕火焰效果
                                if ent == ThePlayer then
                                    ent:PushEvent("fireburn")
                                end
                            end
                            
                            -- 使用计数器控制音效播放，每4次播放一次
                            inst.sound_counter = inst.sound_counter + 1
                            if inst.sound_counter >= 4 then
                                ent.SoundEmitter:PlaySound("dontstarve/common/fireOut")
                                inst.sound_counter = 0 -- 重置计数器
                            end
                        end
                    end
                else
                    -- 如果不再燃烧，停止任务
                    if inst.burn_task then
                        inst.burn_task:Cancel()
                        inst.burn_task = nil
                    end
                end
            end)
        end
    end)
    
    -- 设置熄灭时的处理
    inst.components.burnable:SetOnExtinguishFn(function(inst)
        if inst.fire_fx then
            for i, fx in ipairs(inst.fire_fx) do
                fx:Remove()
            end
            inst.fire_fx = nil
        end
        
        -- 停止伤害任务
        if inst.burn_task then
            inst.burn_task:Cancel()
            inst.burn_task = nil
        end
        
        -- 重置音效计数器
        inst.sound_counter = nil
    end)
    
    -- 设置燃烧完成后的回调函数
    inst.components.burnable:SetOnBurntFn(function(inst)
        -- 移除燃烧特效
        if inst.fire_fx then
            for i, fx in ipairs(inst.fire_fx) do
                fx:Remove()
            end
            inst.fire_fx = nil
        end
        
        -- 停止伤害任务
        if inst.burn_task then
            inst.burn_task:Cancel()
            inst.burn_task = nil
        end
        
        -- 重置音效计数器
        inst.sound_counter = nil
        
        -- 燃烧完成后替换为烧毁版本
        local burnt = SpawnPrefab("potted_boardleaf_burnt")
        if burnt then
            burnt.Transform:SetPosition(inst.Transform:GetWorldPosition())
            burnt.Transform:SetRotation(inst.Transform:GetRotation())
        end
        inst:Remove()
    end)

    -- 配置燃烧属性
    inst.components.burnable:SetBurnTime(10) -- 延长燃烧时间到10秒
    inst.components.burnable.canlight = true -- 允许被点燃
    inst.components.burnable.fxprefab = "fire" -- 设置特效预制体

    -- 添加可燃组件（允许被点燃和传播火焰）
    inst:AddComponent("propagator")
    inst.components.propagator:SetOnFlashPoint(function(inst) 
        -- 当达到闪点时的回调
        if not inst.components.burnable:IsBurning() then
            inst.components.burnable:Ignite()
        end
    end)
    inst.components.propagator.flashpoint = 5 -- 降低闪点温度，更容易点燃
    inst.components.propagator.decayrate = 0.3 -- 降低火焰传播衰减率，燃烧更久
    inst.components.propagator.propagaterange = 3 -- 扩大火焰传播范围
    inst.components.propagator.heatrange = 3 -- 扩大热辐射范围
    inst.components.propagator.damagerange = 3 -- 扩大伤害范围

    -- 添加玩家接近触发器（用于视觉效果）
    inst:AddComponent("playerprox")
    inst.components.playerprox:SetDist(2, 4) -- 扩大触发距离
    inst.components.playerprox:SetOnPlayerNear(function(inst, player)
        if inst.components.burnable and inst.components.burnable:IsBurning() then
            -- 当玩家接近时触发更强的屏幕效果
            player:PushEvent("fireburn")
        end
    end)
    inst.components.playerprox:SetPlayerAliveMode(inst.components.playerprox.AliveModes.AliveOnly)

    -- 添加掉落物组件
    inst:AddComponent("lootdropper")

    -- 设置空的掉落表，因为我们使用自定义的 GenerateLoot 函数
    inst.components.lootdropper:SetLoot({})
    inst.components.lootdropper.numrandomloot = 0

    -- 自定义掉落函数
    inst.components.lootdropper.GenerateLoot = function(self)
        local loot = {}
        
        -- 100%掉落1个蕨叶
        table.insert(loot, "foliage")
        
        -- 50%几率掉落1个石砖
        if math.random() < 0.5 then
            table.insert(loot, "cutstone")
        end
        
        return loot
    end

    -- 可选：如果建筑可以被拆除，添加可摧毁组件。
    inst:AddComponent("workable")
    inst.components.workable:SetWorkAction(ACTIONS.HAMMER) -- 使用锤子拆除
    inst.components.workable:SetWorkLeft(2) -- 需要敲击2下
    -- 设置敲击时的回调
    inst.components.workable:SetOnWorkCallback(function(inst, worker, workleft)
        -- 播放敲击动画
        inst.AnimState:PlayAnimation("potted_boardleaf_hit")
        
        -- 敲击动画播放完后回到闲置状态
        inst:DoTaskInTime(inst.AnimState:GetCurrentAnimationLength(), function()
            if not inst.components.burnable or not inst.components.burnable:IsBurning() then
                inst.AnimState:PlayAnimation("potted_boardleaf_idle", true)
            end
        end)
        
        -- 播放敲击音效（如果有的话）
        inst.SoundEmitter:PlaySound("dontstarve/common/hammer_hit")
        
        -- 如果工作完成（被完全拆除）
        if workleft <= 0 then
            -- 移除燃烧特效
            if inst.fire_fx then
                for i, fx in ipairs(inst.fire_fx) do
                    fx:Remove()
                end
                inst.fire_fx = nil
            end
            
            -- 停止伤害任务
            if inst.burn_task then
                inst.burn_task:Cancel()
                inst.burn_task = nil
            end
            
            -- 重置音效计数器
            inst.sound_counter = nil
            
            -- 播放坍塌特效
            local collapse = SpawnPrefab("collapse_small")
            if collapse then
                collapse.Transform:SetPosition(inst.Transform:GetWorldPosition())
            end
            
            -- 掉落物品
            inst.components.lootdropper:DropLoot()
            
            inst:Remove() -- 移除建筑
        end
    end)

    -- 标记为场景物体，使其在存档时被保存
    inst:AddTag("structure")
    inst:AddTag("potted_boardleaf")

    -- 放置时的初始化
    inst:DoTaskInTime(0, function(inst)
        OnSpawnFn(inst) -- 调用放置特效函数
    end)

    return inst
end

-- 3. 将建筑Prefab添加到游戏中。
return Prefab("potted_boardleaf", fn, assets)